<?php
require __DIR__ . '/../includes/bootstrap.php';
require_login();
$user = current_user();
$pdo = db();

$stmt = $pdo->prepare('SELECT * FROM plans WHERE id = ?');
$stmt->execute([$user['plan_id']]);
$plan = $stmt->fetch();
$maxLinks = (int)($plan['max_links_per_site'] ?? 0);

if (is_post()) {
    if (!csrf_verify()) {
        flash_set('error', 'Invalid session token.');
        redirect('/public/backlinks.php');
    }

    $action = $_POST['action'] ?? '';
    if ($action === 'add') {
        $keyword = trim($_POST['keyword'] ?? '');
        $targetUrl = trim($_POST['target_url'] ?? '');

        if ($keyword === '' || $targetUrl === '') {
            flash_set('error', 'Keyword and URL required.');
        } elseif (!validate_url($targetUrl)) {
            flash_set('error', 'Invalid target URL.');
        } else {
            $stmt = $pdo->prepare('SELECT COUNT(*) FROM backlinks WHERE user_id = ? AND status = "active"');
            $stmt->execute([$user['id']]);
            $count = (int)$stmt->fetchColumn();
            if ($maxLinks > 0 && $count >= $maxLinks) {
                flash_set('error', 'Plan limit reached. Upgrade to add more links.');
            } else {
                $stmt = $pdo->prepare('INSERT INTO backlinks (user_id, keyword, target_url, status, created_at) VALUES (?, ?, ?, "active", NOW())');
                $stmt->execute([$user['id'], $keyword, $targetUrl]);
                log_action((int)$user['id'], 'backlink_add', $keyword . ' => ' . $targetUrl);
                flash_set('success', 'Backlink added.');
            }
        }
    } elseif ($action === 'delete') {
        $id = (int)($_POST['id'] ?? 0);
        $stmt = $pdo->prepare('DELETE FROM backlinks WHERE id = ? AND user_id = ?');
        $stmt->execute([$id, $user['id']]);
        log_action((int)$user['id'], 'backlink_delete', (string)$id);
        flash_set('success', 'Backlink deleted.');
    } elseif ($action === 'toggle') {
        $id = (int)($_POST['id'] ?? 0);
        $stmt = $pdo->prepare('UPDATE backlinks SET status = IF(status = "active", "inactive", "active") WHERE id = ? AND user_id = ?');
        $stmt->execute([$id, $user['id']]);
        log_action((int)$user['id'], 'backlink_toggle', (string)$id);
        flash_set('success', 'Backlink updated.');
    }
    redirect('/public/backlinks.php');
}

$stmt = $pdo->prepare('SELECT * FROM backlinks WHERE user_id = ? ORDER BY created_at DESC');
$stmt->execute([$user['id']]);
$backlinks = $stmt->fetchAll();

$pageTitle = 'Backlinks - XBacklink';
require __DIR__ . '/../includes/header.php';
?>
<div class="row g-4">
    <div class="col-lg-5">
        <div class="card p-4">
            <h4>Add Backlink</h4>
            <form method="post">
                <?php echo csrf_input(); ?>
                <input type="hidden" name="action" value="add">
                <div class="mb-3">
                    <label class="form-label">Anchor Text / Keyword</label>
                    <input class="form-control" name="keyword" required>
                </div>
                <div class="mb-3">
                    <label class="form-label">Target URL</label>
                    <input class="form-control" name="target_url" required>
                </div>
                <button class="btn btn-neon w-100" type="submit">Add Link</button>
            </form>
            <small class="text-muted mt-3 d-block">Limit per plan: <?php echo e((string)$maxLinks); ?></small>
        </div>
    </div>
    <div class="col-lg-7">
        <div class="card p-4">
            <h4>My Backlinks</h4>
            <div class="table-responsive">
                <table class="table table-dark table-striped align-middle">
                    <thead>
                        <tr>
                            <th>Keyword</th>
                            <th>Target</th>
                            <th>Status</th>
                            <th></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($backlinks as $link): ?>
                            <tr>
                                <td><?php echo e($link['keyword']); ?></td>
                                <td><a class="link-muted" href="<?php echo e($link['target_url']); ?>" target="_blank" rel="noopener">View</a></td>
                                <td><span class="badge badge-neon"><?php echo e($link['status']); ?></span></td>
                                <td class="text-end">
                                    <form method="post" class="d-inline">
                                        <?php echo csrf_input(); ?>
                                        <input type="hidden" name="action" value="toggle">
                                        <input type="hidden" name="id" value="<?php echo (int)$link['id']; ?>">
                                        <button class="btn btn-sm btn-outline-light" type="submit">Toggle</button>
                                    </form>
                                    <form method="post" class="d-inline" onsubmit="return confirm('Delete this backlink?');">
                                        <?php echo csrf_input(); ?>
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="id" value="<?php echo (int)$link['id']; ?>">
                                        <button class="btn btn-sm btn-outline-danger" type="submit">Delete</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
<?php require __DIR__ . '/../includes/footer.php'; ?>
