<?php
function e(string $value): string
{
    return htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
}

function redirect(string $url): void
{
    header('Location: ' . $url);
    exit;
}

function is_post(): bool
{
    return ($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST';
}

function random_token(int $length = 32): string
{
    return bin2hex(random_bytes($length));
}

function base_url(): string
{
    $config = require __DIR__ . '/config.php';
    return rtrim($config['app']['base_url'] ?? '', '/');
}

function setting(string $key, ?string $default = null): ?string
{
    static $cache = [];
    if (array_key_exists($key, $cache)) {
        return $cache[$key];
    }
    $pdo = db();
    $stmt = $pdo->prepare('SELECT value FROM settings WHERE `key` = ?');
    $stmt->execute([$key]);
    $value = $stmt->fetchColumn();
    $cache[$key] = $value !== false ? (string)$value : $default;
    return $cache[$key];
}

function flash_set(string $key, string $message): void
{
    $_SESSION['flash'][$key] = $message;
}

function flash_get(string $key): ?string
{
    $message = $_SESSION['flash'][$key] ?? null;
    if ($message !== null) {
        unset($_SESSION['flash'][$key]);
    }
    return $message;
}

function validate_url(string $url): bool
{
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        return false;
    }
    $scheme = parse_url($url, PHP_URL_SCHEME);
    return in_array($scheme, ['http', 'https'], true);
}

function normalize_domain(string $url): string
{
    $host = parse_url($url, PHP_URL_HOST);
    return $host ? strtolower($host) : '';
}

function force_https_if_needed(): void
{
    $config = require __DIR__ . '/config.php';
    if (!($config['app']['force_https'] ?? false)) {
        return;
    }
    $isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || (isset($_SERVER['SERVER_PORT']) && (int)$_SERVER['SERVER_PORT'] === 443);
    if (!$isHttps) {
        $host = $_SERVER['HTTP_HOST'] ?? '';
        $uri = $_SERVER['REQUEST_URI'] ?? '/';
        redirect('https://' . $host . $uri);
    }
}

function log_action(?int $userId, string $action, string $details = ''): void
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO logs (user_id, action, details, ip_address, created_at) VALUES (?, ?, ?, ?, NOW())');
    $stmt->execute([
        $userId,
        $action,
        $details,
        $_SERVER['REMOTE_ADDR'] ?? null,
    ]);
}

function login_rate_limited(string $username): bool
{
    $config = require __DIR__ . '/config.php';
    $window = (int)($config['security']['login_attempt_window_minutes'] ?? 15);
    $max = (int)($config['security']['login_attempt_max'] ?? 5);
    $pdo = db();
    $stmt = $pdo->prepare('SELECT COUNT(*) FROM login_attempts WHERE username = ? AND attempted_at > (NOW() - INTERVAL ? MINUTE)');
    $stmt->execute([$username, $window]);
    $count = (int)$stmt->fetchColumn();
    return $count >= $max;
}

function record_login_attempt(string $username): void
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO login_attempts (username, ip_address, attempted_at) VALUES (?, ?, NOW())');
    $stmt->execute([
        $username,
        $_SERVER['REMOTE_ADDR'] ?? null,
    ]);
}

function clear_login_attempts(string $username): void
{
    $pdo = db();
    $stmt = $pdo->prepare('DELETE FROM login_attempts WHERE username = ?');
    $stmt->execute([$username]);
}
